/*
 * Decompiled with CFR 0.152.
 */
package org.watermedia.api.image;

import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URI;
import java.net.URLConnection;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Iterator;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.function.BiConsumer;
import javax.imageio.ImageIO;
import javax.imageio.ImageReader;
import javax.imageio.metadata.IIOMetadata;
import javax.imageio.stream.ImageInputStream;
import org.w3c.dom.Node;
import org.watermedia.WaterMedia;
import org.watermedia.api.cache.CacheAPI;
import org.watermedia.api.image.ImageAPI;
import org.watermedia.api.image.ImageRenderer;
import org.watermedia.api.image.decoders.GifDecoder;
import org.watermedia.api.network.NetworkAPI;
import org.watermedia.api.network.patchs.AbstractPatch;
import org.watermedia.core.tools.DataTool;
import org.watermedia.core.tools.ThreadTool;

public class ImageFetch
implements Runnable {
    private static final DateFormat FORMAT = new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss z");
    private static final ExecutorService EX = Executors.newScheduledThreadPool(ThreadTool.minThreads(), ThreadTool.factory("ImageFetch-Worker", 6));
    private static final String[] VID_MIMETYPES = new String[]{"video", "audio", "application/vnd.apple.mpegurl", "application/x-mpegurl"};
    public final URI uri;
    public BiConsumer<ImageRenderer, Boolean> successConsumer;
    public BiConsumer<Exception, Boolean> errConsumer;

    public ImageFetch(URI uri) {
        this.uri = uri;
    }

    public ImageFetch setSuccessCallback(BiConsumer<ImageRenderer, Boolean> consumer) {
        this.successConsumer = consumer;
        return this;
    }

    public ImageFetch setErrorCallback(BiConsumer<Exception, Boolean> consumer) {
        this.errConsumer = consumer;
        return this;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void run() {
        try {
            AbstractPatch.Result patch = NetworkAPI.patch(this.uri);
            if (patch == null) {
                throw new IllegalArgumentException("Invalid URL");
            }
            if (patch.assumeVideo) {
                throw new VideoTypeException();
            }
            CacheAPI.Entry cache = CacheAPI.getEntry(this.uri);
            URLConnection conn = null;
            URI patchUri = patch.uri;
            boolean retry = false;
            do {
                try {
                    int code = 200;
                    conn = ImageFetch.openConnection(patchUri, cache);
                    String type = conn.getContentType();
                    if (type != null) {
                        if (DataTool.startsWith(type, VID_MIMETYPES)) {
                            throw new VideoTypeException();
                        }
                        if (!type.startsWith("image")) {
                            throw new NoImageException();
                        }
                    } else {
                        throw new NoImageException();
                    }
                    if (conn instanceof HttpURLConnection) {
                        HttpURLConnection http = (HttpURLConnection)conn;
                        code = http.getResponseCode();
                        switch (code) {
                            case 400: 
                            case 403: 
                            case 404: {
                                throw new ForbiddenException();
                            }
                            case 200: 
                            case 304: {
                                break;
                            }
                            default: {
                                throw new IllegalStateException("HTTP Server responses an invalid status code: " + code);
                            }
                        }
                    }
                    if (cache != null && code == 304) {
                        CacheAPI.updateEntry(new CacheAPI.Entry(patchUri, ImageFetch.getEtagOr(conn, cache.getTag()), ImageFetch.getLastModificationTime(conn), ImageFetch.getExpirationTime(conn)));
                        this.successConsumer.accept(this.readImages(cache), true);
                    } else {
                        InputStream in = conn.getInputStream();
                        byte[] data = DataTool.readAllBytes(in);
                        CacheAPI.saveFile(patchUri, ImageFetch.getEtagOr(conn, cache != null ? cache.getTag() : ""), ImageFetch.getLastModificationTime(conn), ImageFetch.getExpirationTime(conn), data);
                        WaterMedia.LOGGER.debug(ImageAPI.IT, "Successfully downloaded image from '{}'", (Object)patchUri);
                        this.successConsumer.accept(this.readImages(data), false);
                        in.close();
                    }
                    retry = false;
                }
                catch (Exception e) {
                    if (e instanceof ForbiddenException) {
                        AbstractPatch.Result result = patch.fallbackResult.compute(this.uri);
                        if (result != null) {
                            patchUri = result.uri;
                            retry = true;
                            continue;
                        }
                        patchUri = null;
                    }
                    if (cache == null || !cache.getFile().exists()) {
                        throw e;
                    }
                    WaterMedia.LOGGER.error(ImageAPI.IT, "Failed to fetch image, delegating to cache files");
                    this.successConsumer.accept(this.readImages(cache), true);
                }
                finally {
                    if (conn instanceof HttpURLConnection) {
                        ((HttpURLConnection)conn).disconnect();
                    }
                }
            } while (retry);
        }
        catch (InternalDecoderException | NoImageException e) {
            WaterMedia.LOGGER.error(ImageAPI.IT, "Invalid image source from '{}'", (Object)this.uri, (Object)e);
            this.errConsumer.accept(e, false);
        }
        catch (VideoTypeException e) {
            WaterMedia.LOGGER.debug(ImageAPI.IT, "Detected a video type from '{}'", (Object)this.uri);
            this.errConsumer.accept(e, true);
        }
        catch (Exception e) {
            WaterMedia.LOGGER.error(ImageAPI.IT, "Unhandled exception occurred while loading image from '{}'", (Object)this.uri, (Object)e);
            this.errConsumer.accept(e, false);
        }
        catch (Throwable t) {
            WaterMedia.LOGGER.error(ImageAPI.IT, "Fatal exception occurred while loading image from '{}'", (Object)this.uri, (Object)t);
            this.errConsumer.accept(new Exception("Fatal exception running image loading", t), false);
        }
    }

    public void start() {
        EX.execute(this);
    }

    public ImageRenderer readImages(CacheAPI.Entry cache) throws Exception {
        try (InputStream in = Files.newInputStream(cache.getFile().toPath(), new OpenOption[0]);){
            byte[] data = DataTool.readAllBytes(in);
            ImageRenderer imageRenderer = this.readImages(data);
            return imageRenderer;
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public ImageRenderer readImages(byte[] data) throws Exception {
        String type = "";
        try (ImageInputStream stream = ImageIO.createImageInputStream(new ByteArrayInputStream(data));){
            long[] delay;
            BufferedImage[] images;
            block17: {
                Iterator<ImageReader> iterator = ImageIO.getImageReaders(stream);
                while (iterator.hasNext()) {
                    ImageReader reader = iterator.next();
                    reader.setInput(stream, false, false);
                    int frames = reader.getNumImages(true);
                    images = new BufferedImage[frames];
                    delay = new long[frames];
                    long lastDelay = 10L;
                    int noDelayFrames = 0;
                    try {
                        for (int i = 0; i < frames; ++i) {
                            IIOMetadata metadata = reader.getImageMetadata(i);
                            String format = metadata.getNativeMetadataFormatName();
                            type = reader.getFormatName();
                            if (reader.getFormatName().equalsIgnoreCase("gif")) {
                                throw new IllegalArgumentException("Gif format detected, forced-delegation to WM decoders");
                            }
                            images[i] = reader.read(i);
                        }
                        if (noDelayFrames > 0) {
                            WaterMedia.LOGGER.debug(ImageAPI.IT, "Gif decoder reports {} frames without delaay", (Object)noDelayFrames);
                        }
                        break block17;
                    }
                    catch (Exception e) {
                        if (type.equalsIgnoreCase("gif")) {
                            throw e;
                        }
                        WaterMedia.LOGGER.error(ImageAPI.IT, "Failed to decode image using reader {}({})", (Object)reader.getClass().getSimpleName(), (Object)reader.getFormatName());
                        WaterMedia.LOGGER.debug(ImageAPI.IT, "Error: ", (Throwable)e);
                    }
                }
                throw new IOException("ImageFetcher was unable to read the image");
            }
            ImageRenderer imageRenderer = ImageAPI.renderer(images, delay);
            return imageRenderer;
        }
        catch (Exception e) {
            if (!type.equalsIgnoreCase("gif")) {
                throw e;
            }
            WaterMedia.LOGGER.error(ImageAPI.IT, "Failed to decode gif via ImageIO, delegating to WaterMedia decoders");
            WaterMedia.LOGGER.debug(ImageAPI.IT, "Error: ", (Throwable)e);
            GifDecoder gif = new GifDecoder();
            int status = gif.read(data);
            if (status != 0) throw new InternalDecoderException("Failed to decode gif, status code: " + status);
            return ImageAPI.renderer(gif);
        }
    }

    private static String getEtagOr(URLConnection conn, String alt) {
        String ETag = conn.getHeaderField("ETag");
        if (ETag == null || ETag.isEmpty()) {
            return alt;
        }
        return ETag;
    }

    private static long getExpirationTime(URLConnection conn) {
        long parsed;
        long time;
        String maxAge = conn.getHeaderField("max-age");
        if (maxAge == null || maxAge.isEmpty()) {
            maxAge = "-1";
        }
        long l = time = (parsed = DataTool.parseLongOr(maxAge, -1L)) == -1L ? -1L : System.currentTimeMillis() + parsed * 100L;
        if (time != -1L) {
            return time;
        }
        String expires = conn.getHeaderField("Expires");
        if (expires != null && !expires.isEmpty()) {
            try {
                time = FORMAT.parse(expires).getTime();
            }
            catch (NumberFormatException | ParseException exception) {
                // empty catch block
            }
        }
        return time;
    }

    private static long getLastModificationTime(URLConnection conn) {
        long time = -1L;
        String date = conn.getHeaderField("Last-Modified");
        if (date != null && !date.isEmpty()) {
            try {
                time = FORMAT.parse(date).getTime();
            }
            catch (NumberFormatException | ParseException exception) {
                // empty catch block
            }
        }
        return time;
    }

    private static Node fetchImageNode(Node root, String nodeName) {
        if (root.getNodeName().equalsIgnoreCase(nodeName)) {
            return root;
        }
        for (Node child = root.getFirstChild(); child != null; child = child.getNextSibling()) {
            Node result = ImageFetch.fetchImageNode(child, nodeName);
            if (result == null) continue;
            return result;
        }
        return null;
    }

    private static URLConnection openConnection(URI uri, CacheAPI.Entry cache) throws IOException {
        URLConnection conn = uri.toURL().openConnection();
        conn.setDefaultUseCaches(false);
        conn.setRequestProperty("Accept", "image/*");
        if (cache != null && cache.getFile().exists()) {
            if (cache.getTag() != null) {
                conn.setRequestProperty("If-None-Match", cache.getTag());
            } else if (cache.getTime() != -1L) {
                conn.setRequestProperty("If-Modified-Since", FORMAT.format(new Date(cache.getTime())));
            }
        }
        return conn;
    }

    private static class VideoTypeException
    extends Exception {
        private VideoTypeException() {
        }
    }

    private static class NoImageException
    extends Exception {
        private NoImageException() {
        }
    }

    private static class ForbiddenException
    extends NoImageException {
        private ForbiddenException() {
        }
    }

    private static class InternalDecoderException
    extends Exception {
        public InternalDecoderException(String msg) {
            super(msg);
        }
    }
}

